import React, { useEffect, useRef, useState } from "react";
import confetti from "canvas-confetti";
import { usePuzzleStore } from "../store/puzzleStore";

export default function PuzzleCard({ question, onNext }) {
  const { markCorrect, markWrong } = usePuzzleStore();
  const [selected, setSelected] = useState(null);
  const [locked, setLocked] = useState(false);
  const [timer, setTimer] = useState(question?.timeLimit ?? 20);
  const [overlay, setOverlay] = useState(null);
  const timerRef = useRef(null);
  const cardRef = useRef(null);

  // Reset state when new question arrives
  useEffect(() => {
    clearInterval(timerRef.current);
    setSelected(null);
    setLocked(false);
    setOverlay(null);
    setTimer(question?.timeLimit ?? 20);
    startTimer();
    return () => clearInterval(timerRef.current);
  }, [question?.id]);

  const startTimer = () => {
    clearInterval(timerRef.current);
    timerRef.current = setInterval(() => {
      setTimer((prev) => {
        if (prev <= 1) {
          clearInterval(timerRef.current);
          handleTimesUp();
          return 0;
        }
        return prev - 1;
      });
    }, 1000);
  };

  const flashWrong = () => {
    if (!cardRef.current) return;
    cardRef.current.classList.add("ring-2", "ring-red-500");
    setTimeout(() => {
      cardRef.current?.classList.remove("ring-2", "ring-red-500");
    }, 400);
  };

  const celebrate = () => {
    try {
      confetti({ particleCount: 70, spread: 60, origin: { y: 0.3 } });
    } catch {}
  };

  // 🧩 Move safely to next question
  const goNext = (wasCorrect, timeTaken, delay = 1500) => {
    clearInterval(timerRef.current);
    setTimeout(() => {
      if (typeof onNext === "function") {
        onNext(wasCorrect, timeTaken);
      }
      // Reset overlay AFTER next question loads
      setOverlay(null);
      setLocked(false);
      setSelected(null);
    }, delay);
  };

  // ⏰ Times Up
  const handleTimesUp = () => {
    if (locked) return;
    setLocked(true);
    const timeTaken = question.timeLimit ?? 20;
    setOverlay({ msg: "⏰ Time’s Up!", color: "text-yellow-400" });
    flashWrong();
    markWrong(timeTaken);
    goNext(false, timeTaken, 1500);
  };

  // ✅ Submit Answer
  const handleSubmit = () => {
    if (locked || selected == null) return;
    clearInterval(timerRef.current);
    setLocked(true);

    const timeTaken = (question.timeLimit ?? 20) - timer;

    if (selected === question.answer) {
      markCorrect(timeTaken);
      celebrate();
      setOverlay({ msg: "✅ Correct!", color: "text-green-400" });
      goNext(true, timeTaken, 1000);
    } else {
      markWrong(timeTaken);
      flashWrong();
      setOverlay({ msg: "❌ Wrong!", color: "text-red-400" });
      goNext(false, timeTaken, 1200);
    }
  };

  return (
    <div
      ref={cardRef}
      className="relative glass p-6 rounded-2xl card-shadow w-full overflow-hidden transition-all"
    >
      {/* Overlay */}
      {overlay && (
        <div className="absolute inset-0 flex items-center justify-center bg-black/70 z-20 pointer-events-none">
          <div className={`text-3xl font-bold ${overlay.color}`}>
            {overlay.msg}
          </div>
        </div>
      )}

      {/* Header */}
      <div className="flex items-start justify-between">
        <div>
          <p className="text-sm text-white/70">{question?.title}</p>
          <h2 className="text-2xl font-semibold mt-1">{question?.question}</h2>
        </div>
        <div className="text-sm bg-white/10 px-3 py-1 rounded-full text-white/90">
          ⏳ {timer}s
        </div>
      </div>

      {/* Options */}
      <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 mt-5">
        {question?.options?.map((opt, i) => (
          <button
            key={i}
            onClick={() => setSelected(opt)}
            disabled={locked}
            className={`py-3 px-4 rounded-lg border border-white/10 text-left transition-all ${
              selected === opt
                ? "bg-indigo-600 text-white"
                : "bg-white/5 hover:bg-white/15"
            }`}
          >
            {opt}
          </button>
        ))}
      </div>

      {/* Footer */}
      <div className="flex items-center justify-between mt-6">
        <p className="text-sm text-white/70">Tip: Submit before time ends</p>

        <div className="flex gap-3">
          <button
            onClick={() => setSelected(null)}
            disabled={locked}
            className="px-4 py-2 bg-white/10 rounded"
          >
            Clear
          </button>

          <button
            onClick={handleSubmit}
            disabled={!selected || locked}
            className="px-4 py-2 bg-gradient-to-r from-cyan-400 to-violet-500 rounded text-black font-semibold"
          >
            Submit
          </button>
        </div>
      </div>
    </div>
  );
}
